import { BufferedBlockAlgorithm } from './BufferedBlockAlgorithm';
import { WordArray } from './WordArray';
import { BufferedBlockAlgorithmConfig } from './BufferedBlockAlgorithmConfig';
import { CipherParams } from './CipherParams';
export declare abstract class Cipher extends BufferedBlockAlgorithm {
    /**
     * A constant representing encryption mode.
     */
    static _ENC_XFORM_MODE: number;
    /**
     * A constant representing decryption mode.
     */
    static _DEC_XFORM_MODE: number;
    /**
     * This cipher's key size. Default: 4 (128 bits / 32 Bits)
     */
    static keySize: number;
    /**
     * This cipher's IV size. Default: 4 (128 bits / 32 Bits)
     */
    static ivSize: number;
    /**
     * Either the encryption or decryption transformation mode constant.
     */
    _xformMode: number;
    /**
     * The key.
     */
    _key: WordArray;
    /**
     * Creates this cipher in encryption mode.
     *
     * @param key The key.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @return A cipher instance.
     *
     * @example
     *
     *     let cipher = AES.createEncryptor(keyWordArray, { iv: ivWordArray });
     */
    static createEncryptor(key: WordArray, cfg?: BufferedBlockAlgorithmConfig): Cipher;
    /**
     * Creates this cipher in decryption mode.
     *
     * @param key The key.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @return A cipher instance.
     *
     * @example
     *
     *     let cipher = AES.createDecryptor(keyWordArray, { iv: ivWordArray });
     */
    static createDecryptor(key: WordArray, cfg?: BufferedBlockAlgorithmConfig): Cipher;
    /**
     * Creates shortcut functions to a cipher's object interface.
     *
     * @param cipher The cipher to create a helper for.
     *
     * @return An object with encrypt and decrypt shortcut functions.
     *
     * @example
     *
     *     let AES = Cipher._createHelper(AESAlgorithm);
     */
    static _createHelper(cipher: typeof Cipher): {
        encrypt: (message: string | WordArray, key: string | WordArray, cfg?: BufferedBlockAlgorithmConfig | undefined) => CipherParams;
        decrypt: (ciphertext: string | CipherParams, key: string | WordArray, cfg?: BufferedBlockAlgorithmConfig | undefined) => WordArray;
    };
    /**
     * Initializes a newly created cipher.
     *
     * @param xformMode Either the encryption or decryption transormation mode constant.
     * @param key The key.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @example
     *
     *     let cipher = AES.create(AES._ENC_XFORM_MODE, keyWordArray, { iv: ivWordArray });
     */
    constructor(xformMode: number, key: WordArray, cfg?: BufferedBlockAlgorithmConfig);
    /**
     * Adds data to be encrypted or decrypted.
     *
     * @param dataUpdate The data to encrypt or decrypt.
     *
     * @return The data after processing.
     *
     * @example
     *
     *     let encrypted = cipher.process('data');
     *     let encrypted = cipher.process(wordArray);
     */
    process(dataUpdate: WordArray | string): WordArray;
    /**
     * Finalizes the encryption or decryption process.
     * Note that the finalize operation is effectively a destructive, read-once operation.
     *
     * @param dataUpdate The final data to encrypt or decrypt.
     *
     * @return The data after final processing.
     *
     * @example
     *
     *     var encrypted = cipher.finalize();
     *     var encrypted = cipher.finalize('data');
     *     var encrypted = cipher.finalize(wordArray);
     */
    finalize(dataUpdate?: WordArray | string): WordArray;
    /**
     * Cipher specific finalize function explicitly implemented in the derived class.
     */
    abstract _doFinalize(): WordArray;
}
