import { WordArray } from './WordArray';
import { Cipher } from './Cipher';
import { BufferedBlockAlgorithmConfig } from './BufferedBlockAlgorithmConfig';
import { CipherParams } from './CipherParams';
import { Formatter } from '../format/Formatter';
export declare class PasswordBasedCipher {
    static cfg: BufferedBlockAlgorithmConfig;
    /**
     * Encrypts a message using a password.
     *
     * @param cipher The cipher algorithm to use.
     * @param message The message to encrypt.
     * @param password The password.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @return A cipher params object.
     *
     * @example
     *
     *     var ciphertextParams = CryptoJS.lib.PasswordBasedCipher.encrypt(AES, message, 'password');
     *     var ciphertextParams = CryptoJS.lib.PasswordBasedCipher.encrypt(AES, message, 'password', { format: OpenSSL });
     */
    static encrypt(cipher: typeof Cipher, message: WordArray | string, password: string, cfg?: BufferedBlockAlgorithmConfig): CipherParams;
    /**
     * Decrypts serialized ciphertext using a password.
     *
     * @param cipher The cipher algorithm to use.
     * @param ciphertext The ciphertext to decrypt.
     * @param password The password.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @return The plaintext.
     *
     * @example
     *
     *     var plaintext = PasswordBasedCipher.decrypt(AES, formattedCiphertext, 'password', { format: OpenSSL });
     *     var plaintext = PasswordBasedCipher.decrypt(AES, ciphertextParams, 'password', { format: OpenSSL });
     */
    static decrypt(cipher: typeof Cipher, ciphertext: CipherParams | string, password: string, cfg?: BufferedBlockAlgorithmConfig): WordArray;
    /**
     * Converts serialized ciphertext to CipherParams,
     * else assumed CipherParams already and returns ciphertext unchanged.
     *
     * @param ciphertext The ciphertext.
     * @param format The formatting strategy to use to parse serialized ciphertext.
     *
     * @return The unserialized ciphertext.
     *
     * @example
     *
     *     var ciphertextParams = CryptoJS.lib.SerializableCipher._parse(ciphertextStringOrParams, format);
     */
    static _parse(ciphertext: CipherParams | string, format: Formatter): CipherParams;
}
