import { WordArray } from './WordArray';
import { Cipher } from './Cipher';
import { BufferedBlockAlgorithmConfig } from './BufferedBlockAlgorithmConfig';
import { CipherParams } from './CipherParams';
import { Formatter } from '../format/Formatter';
export declare class SerializableCipher {
    static cfg: BufferedBlockAlgorithmConfig;
    /**
     * Encrypts a message.
     *
     * @param cipher The cipher algorithm to use.
     * @param message The message to encrypt.
     * @param key The key.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @return A cipher params object.
     *
     * @example
     *
     *     let ciphertextParams = SerializableCipher.encrypt(CryptoJS.algo.AES, message, key);
     *     let ciphertextParams = SerializableCipher.encrypt(CryptoJS.algo.AES, message, key, { iv: iv });
     *     let ciphertextParams = SerializableCipher.encrypt(CryptoJS.algo.AES, message, key, {
     *       iv: iv,
     *       format: CryptoJS.format.OpenSSL
     *     });
     */
    static encrypt(cipher: typeof Cipher, message: WordArray | string, key: WordArray, cfg?: BufferedBlockAlgorithmConfig): CipherParams;
    /**
     * Decrypts serialized ciphertext.
     *
     * @param cipher The cipher algorithm to use.
     * @param ciphertext The ciphertext to decrypt.
     * @param key The key.
     * @param cfg (Optional) The configuration options to use for this operation.
     *
     * @return The plaintext.
     *
     * @example
     *
     *     let plaintext = SerializableCipher.decrypt(
     *         AESAlgorithm,
     *         formattedCiphertext,
     *         key, {
     *             iv: iv,
     *             format: CryptoJS.format.OpenSSL
     *         }
     *     );
     *
     *     let plaintext = SerializableCipher.decrypt(
     *         AESAlgorithm,
     *         ciphertextParams,
     *         key, {
     *             iv: iv,
     *             format: CryptoJS.format.OpenSSL
     *         }
     *     );
     */
    static decrypt(cipher: typeof Cipher, ciphertext: CipherParams | string, key: WordArray, optionalCfg?: BufferedBlockAlgorithmConfig): WordArray;
    /**
     * Converts serialized ciphertext to CipherParams,
     * else assumed CipherParams already and returns ciphertext unchanged.
     *
     * @param ciphertext The ciphertext.
     * @param format The formatting strategy to use to parse serialized ciphertext.
     *
     * @return The unserialized ciphertext.
     *
     * @example
     *
     *     var ciphertextParams = CryptoJS.lib.SerializableCipher._parse(ciphertextStringOrParams, format);
     */
    static _parse(ciphertext: CipherParams | string, format: Formatter): CipherParams;
}
